/* htable.c
 * 
 * Copyright (C) 2005 2006 Toon Calders, Bart Goethals, Szymon Jaroszewicz
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "htable.h"


static void
htable_alloc_buckets(htable * h, size_t alloc_size)
{
    size_t i;

    h -> alloc_size = alloc_size;
    h -> entries = (entry *) malloc(h->alloc_size * sizeof(entry));
    if(h -> entries == NULL)
    {
	return;
    }
    for(i = 0; i < h->alloc_size; i++)
    {
	h->entries[i].key = NULL;
	h->entries[i].value = NULL;
    }
    return;
}

htable *
htable_new(size_t (* hashfunc)(void *), int (* cmpfunc)(void *, void *))
{
    htable * h = (htable *) malloc(sizeof(htable));
//    size_t i;

    if(h != NULL)
    {
	h -> hashfunc = hashfunc;
	h -> cmpfunc = cmpfunc;
	h -> n = 0;
	htable_alloc_buckets(h, 17);
	if(h -> entries == NULL)
	{
	    free(h);
	    h = NULL;
	}
    }
    return h;
}

void
htable_free(htable * h, int free_keys, int free_values)
{
    size_t i;

    for(i = 0; i < h->alloc_size; i++)
    {
	if(h->entries[i].key != NULL)
	{
	    if(free_keys) free(h->entries[i].key);
	    if(free_values) free(h->entries[i].value);
	}
    }
    free(h->entries);
    free(h);
}

static size_t get_bucket(htable * h, void * key)
{
    size_t hash = h->hashfunc(key) % h->alloc_size;
	
    while(h->entries[hash].key != NULL && !h->cmpfunc(key, h->entries[hash].key))
    {
		hash = (hash + 1) % h->alloc_size;
    }
    return hash;
}

void *
htable_get(htable * h, void * key)
{
    size_t hash = get_bucket(h, key);
    void * value = NULL;

    if(h->entries[hash].key != NULL)
    {
	value = h->entries[hash].value;
    }
    return value;
}

static void
htable_rehash(htable * h)
{
    size_t old_alloc_size = h->alloc_size;
    entry * oldentries = h->entries;
    size_t i;

    //printf("rehash\n");
    htable_alloc_buckets(h, 2 * old_alloc_size + 1);
    if(h->entries == NULL)
    {
	;// should report error
    }
    h->n = 0;
    for(i = 0; i < old_alloc_size; i++)
    {
	if(oldentries[i].key != NULL)
	{
	    htable_put(h, oldentries[i].key, oldentries[i].value);
	}
    }
    free(oldentries);
}

void
htable_put(htable * h, void * key, void * value)
{
    size_t hash;
	
    if(2*h->n >= h->alloc_size)
    {
		htable_rehash(h);
    }
    hash = get_bucket(h, key);
    h->entries[hash].key = key;
    h->entries[hash].value = value;
    h->n++;
}
