/* range_tree.c
 * 
 * Copyright (C) 2005 2006 Toon Calders, Bart Goethals, Szymon Jaroszewicz
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "range_tree.h"

//new
void my_print_rt( range_tree *r)
{
	int s;
	for(s = 0; s < r->listlen; s++)
	{
		printf("%f (%d,%d)", r->list[s].value, r->list[s].counter, r->list[s].counterGrt);
		if(s < r->listlen - 1)
				printf(",");
	}
	printf("]\n");
}


static int
compare_ind_entries(const void * a_ptr, const void * b_ptr)
{
    double a = ((index_entry *)a_ptr)->value;
    double b = ((index_entry *)b_ptr)->value;

    if(isnan(a) && isnan(b)) return 0;
    if(isnan(a)) return -1;
    if(isnan(b)) return 1;
    if(a < b) return -1;
    if(a == b) return 0;
    return 1;
}

index_entry **
range_tree_create_indices(dataset * d)
{
    index_entry ** indices;
    size_t i, j;
	
    indices = (index_entry **) malloc(sizeof(index_entry *) * d->n_items);
    if(indices != NULL)
    {
		// allocate memory
 		for(i = 0; i < d->n_items; i++)
		{
			indices[i] = (index_entry *) malloc(d->n * sizeof(index_entry));
			if(indices[i] == NULL)
			{
				for(j = 0; j < i; j++)
				{
					free(indices[i]);
				}
				free(indices);
				return NULL;
			}
		}
		// copy data
		for(i = 0; i < d->n_items; i++) //for every attribute i
		{
			for(j = 0; j < d->n; j++) //for every row j in the dataset
			{
				indices[i][j].record = d->data[j];
				indices[i][j].value = d->data[j][i];
			}
		}
		// sort
		for(i = 0; i < d->n_items; i++)
		{
			qsort(indices[i], d->n, sizeof(index_entry), compare_ind_entries);
		}
    }
    return indices;
}

void
range_tree_free_indices(index_entry ** indices, size_t n_indices)
{
    size_t i;
	
    for(i = 0; i < n_indices; i++)
    {
		free(indices[i]);
    }
    free(indices);
}




static void
range_tree_init(range_tree * r, double value)
{
    r->counter = 1;
    r->list = NULL;
    r->listlen = 0;
    r->value = value;
    //r->dim = ?;
}

range_tree *
range_tree_new(double value)
{
    range_tree * r = (range_tree *) malloc(sizeof(range_tree));
    if(r != NULL)
    {
		range_tree_init(r, value);
    }
    return r;
}

static void
range_tree_free_rec(range_tree * r)
{
    size_t i;

    if(r->list != NULL)
    {
		for(i = 0; i < r->listlen; i++)
		{
			range_tree_free_rec(r->list + i);
		}
		free(r->list);
    }
}

void
range_tree_free(range_tree * r)
{
    range_tree_free_rec(r);
    free(r);
}

static size_t 
compute_counts(range_tree * r, size_t beg, size_t n)
{
    size_t n2 = n / 2;
    size_t i = beg + n2;
    size_t tot, tot2;
	
    if(n == 0)
    {
		return 0;
    }
    tot = compute_counts(r, beg, n2);
    tot2 = compute_counts(r, i + 1, n - n2 - 1);
    r->list[i].counter += tot; 
    return r->list[i].counter + tot2; 
}

//new
static size_t 
compute_countsGrt(range_tree * r, size_t beg, size_t n)
{
    size_t n2 = n / 2;
    size_t i = beg + n2;
    size_t tot, tot2;
	
    if(n == 0)
    {
		return 0;
    }
    tot = compute_countsGrt(r, i + 1, n - n2 - 1);
    tot2 = compute_countsGrt(r, beg, n2);
    r->list[i].counterGrt += tot; 
    return r->list[i].counterGrt + tot2; 
}



static void
add_dim_rec(range_tree * r, size_t a, index_entry * ind, size_t indlen,  size_t beg, size_t n )
{
	//ind contains the rows for every value of the dimension a

    size_t n2 = n / 2;
    size_t i = beg + n2;
    size_t cnt1 = 0, cnt2 = 0, cnt3 = 0;
    size_t s, s1, s2, s3;
    index_entry * left, * left2, * right;
	
    if(n == 0)
    {
		return;
    }
    for(s = 0; s < indlen; s++) //for every row 
    {
		if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <  r->list[i].value) 
			cnt1++;
		if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <= r->list[i].value) 
			cnt2++;
		if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] >  r->list[i].value) 
			cnt3++;
    }
    left  = (index_entry *) malloc(sizeof(index_entry) * cnt1);
    left2 = (index_entry *) malloc(sizeof(index_entry) * cnt2);
    right = (index_entry *) malloc(sizeof(index_entry) * cnt3);

    s1 = s2 = s3 = 0;
    for(s = 0; s < indlen; s++)
    {
		if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <  r->list[i].value) 
			left [s1++] = ind[s]; 
		if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <= r->list[i].value) 
			left2[s2++] = ind[s];
		if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] >  r->list[i].value) 
			right[s3++] = ind[s];
    }

    range_tree_add_dim(r->list +i, a, left2, cnt2);

    if(n2 > 0)
		add_dim_rec(r, a, left,  cnt1, beg, n2);
    
	if(n > n2 + 1)
		add_dim_rec(r, a, right, cnt3, i + 1, n - n2 - 1);
    
	free(right);
    free(left2);
    free(left);
}

//new
static void
my_add_dim_rec(range_tree * r, size_t a, index_entry * ind, size_t indlen,  size_t beg, size_t n, unsigned short int is_positive)
{
	//ind contains the rows for every value of the dimension a
	
    size_t n2 = n / 2;
    size_t i = beg + n2;
    size_t cnt1 = 0, cnt2 = 0, cnt3 = 0;
    size_t s, s1, s2, s3;
    index_entry * left, * left2, * right;
	
    if(n == 0)
    {
		return;
    }
	
	if (r->is_positive != 0)
	{
		
		for(s = 0; s < indlen; s++) //for every row 
		{
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <  r->list[i].value) 
				cnt1++;
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <= r->list[i].value) 
				cnt2++;
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] >  r->list[i].value) 
				cnt3++;
		}
		left  = (index_entry *) malloc(sizeof(index_entry) * cnt1);
		left2 = (index_entry *) malloc(sizeof(index_entry) * cnt2);
		right = (index_entry *) malloc(sizeof(index_entry) * cnt3);
		
		s1 = s2 = s3 = 0;
		for(s = 0; s < indlen; s++)
		{
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <  r->list[i].value) 
				left [s1++] = ind[s]; 
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <= r->list[i].value) 
				left2[s2++] = ind[s];
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] >  r->list[i].value) 
				right[s3++] = ind[s];
		}
		
		my_range_tree_add_dim(r->list +i, a, left2, cnt2, is_positive);
		
		if(n2 > 0)
			my_add_dim_rec(r, a, left,  cnt1, beg, n2, is_positive);
		
		if(n > n2 + 1)
			my_add_dim_rec(r, a, right, cnt3, i + 1, n - n2 - 1, is_positive);

	}
	else // negative
	{
		
		for(s = 0; s < indlen; s++) //for every row 
		{
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] >  r->list[i].value) 
				cnt1++;
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] >= r->list[i].value) 
				cnt2++;
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <  r->list[i].value) 
				cnt3++;
		}
		left  = (index_entry *) malloc(sizeof(index_entry) * cnt1);
		left2 = (index_entry *) malloc(sizeof(index_entry) * cnt2);
		right = (index_entry *) malloc(sizeof(index_entry) * cnt3);
		
		s1 = s2 = s3 = 0;
		for(s = 0; s < indlen; s++)
		{
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] >  r->list[i].value) 
				left [s1++] = ind[s]; 
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] >= r->list[i].value) 
				left2[s2++] = ind[s];
			if(!isnan(ind[s].record[r->dim]) && ind[s].record[r->dim] <  r->list[i].value) 
				right[s3++] = ind[s];
		}
		
		my_range_tree_add_dim(r->list +i, a, left2, cnt2, is_positive);
		
		if(n2 > 0)
			my_add_dim_rec(r, a, left,  cnt1, i + 1, n - n2 - 1, is_positive);
		
		if(n > n2 + 1)
			my_add_dim_rec(r, a, right, cnt3, beg, n2, is_positive);
		
	}
	free(right);
	free(left2);
	free(left);
}


void range_tree_add_dim(range_tree * r, size_t a, index_entry * ind, size_t indlen)
{
    double x, old_x;
	//    size_t i;
    size_t cnt = 0;
    size_t counter;
    size_t s;
	
    if(indlen == 0)
		return;
    if(r->list == NULL) 
    {
		r->dim = a;

		// count list size
		old_x = ind[0].value; 
		cnt = 0;
		for(s = 0; s < indlen; s++)
		{
			x = ind[s].value;
			//if((isnan(old_x) && !isnan(x)) || x != old_x)
			if(x != old_x)
			{
				if(!isnan(old_x))
				{
					cnt++;
				}
				old_x = x;
			}
		}
		if(!isnan(old_x))
		{
			cnt++;
		}
		r->listlen = cnt; //there are cnt different rank values for the dimension dim=a
		
		// create the list
		r->list = (range_tree *)malloc(sizeof(range_tree) * r->listlen);
		old_x = ind[0].value;
		cnt = 0;
		counter = 0;
		for(s = 0; s < indlen; s++)
		{
			x = ind[s].value;
			//if((isnan(old_x) && !isnan(x)) || x != old_x)
			if(x != old_x)
			{
				if(!isnan(old_x))
				{
					range_tree_init(r->list + cnt, old_x);
					r->list[cnt].counter = counter;
					r->list[cnt].counterGrt = counter; //new
					cnt++;
				}
				counter = 1;
				old_x = x;
			}
			else
			{
				counter++;
			}
		}
		if(!isnan(old_x))
		{
			range_tree_init(r->list + cnt, old_x);
			r->list[cnt].counter = counter;
		}
		
		compute_counts(r, 0, r->listlen);
    }
    else
    {
		add_dim_rec(r, a, ind, indlen, 0, r->listlen);
    }
}


//new
void 
my_range_tree_add_dim(range_tree * r, size_t a, index_entry * ind, size_t indlen, unsigned short int is_positive)
{
    double x, old_x;
	//    size_t i;
    size_t cnt = 0;
    size_t counter;
    size_t s;
	
    if(indlen == 0)
		return;
    if(r->list == NULL) 
    {
		r->dim = a;
		r->is_positive = is_positive;

		// count list size
		old_x = ind[0].value; 
		cnt = 0;
		for(s = 0; s < indlen; s++)
		{
			x = ind[s].value;
			//if((isnan(old_x) && !isnan(x)) || x != old_x)
			if(x != old_x)
			{
				if(!isnan(old_x))
				{
					cnt++;
				}
				old_x = x;
			}
		}
		if(!isnan(old_x))
		{
			cnt++;
		}
		r->listlen = cnt; //there are cnt different rank values for the dimension dim=a
		
		// create the list
		r->list = (range_tree *)malloc(sizeof(range_tree) * r->listlen);
		old_x = ind[0].value;
		cnt = 0;
		counter = 0;
		for(s = 0; s < indlen; s++)
		{
			x = ind[s].value;
			//if((isnan(old_x) && !isnan(x)) || x != old_x)
			if(x != old_x)
			{
				if(!isnan(old_x))
				{
					range_tree_init(r->list + cnt, old_x);
					r->list[cnt].counter = counter;
					r->list[cnt].counterGrt = counter; //new
					cnt++;
				}
				counter = 1;
				old_x = x;
			}
			else
			{
				counter++;
			}
		}
		if(!isnan(old_x))
		{
			range_tree_init(r->list + cnt, old_x);
			r->list[cnt].counter = counter;
			r->list[cnt].counterGrt = counter; //new
		}
		
		compute_counts(r, 0, r->listlen);
		compute_countsGrt(r, 0, r->listlen); //new
    }
    else
    {
		my_add_dim_rec(r, a, ind, indlen, 0, r->listlen, is_positive);
    }
}





static size_t
get_count_rec(range_tree * r, double * bounds, size_t lenbounds, size_t beg, size_t n)
{


    size_t n2 = n / 2;
    size_t i = beg + n2;
    //double v;
	double node;
    size_t ret = 0;
	
    if(n == 0)
    {
		return 0;
    }
    node = r->list[i].value;
	
    //if(!isnan(v) && !isnan(bounds[0]) && v < bounds[0])
    if(node < bounds[0])
    {
		ret = range_tree_get_count(r->list + i, bounds + 1, lenbounds - 1); //go inside each dimension range tree
		ret += get_count_rec(r, bounds, lenbounds, i + 1, n - n2 - 1); //go to the right side!! 
    }
    //else if(!isnan(v) && !isnan(bounds[0]) && v == bounds[0])
    else if(node == bounds[0])
    {
		ret = get_count_rec(r, bounds, lenbounds, beg, n2); //have a look at the left side
    }
    //else if(!isnan(v) && !isnan(bounds[0]) && v > bounds[0])
    else if(node > bounds[0])
    {
		ret = get_count_rec(r, bounds, lenbounds, beg, n2);
    }
    return ret;
}


static size_t
my_get_count_rec(range_tree * r, Bound * bounds, size_t lenbounds, size_t beg, size_t n)
{
    size_t n2 = n / 2;
    size_t i = beg + n2;
    //double v;
	double node;
    size_t ret = 0;

    if(n == 0)
    {
		return 0;
    }
    node = r->list[i].value;
	
    if (bounds[0].is_positive != 0)
    {
		
		//if(!isnan(v) && !isnan(bounds[0]) && v < bounds[0])
		if(node < bounds[0].value)
		{
			ret = my_range_tree_get_count(r->list + i, bounds + 1, lenbounds - 1, bounds[0].is_positive); //go to the range tree of the following attribute
			ret += my_get_count_rec(r, bounds, lenbounds, i + 1, n - n2 - 1); //go to the right side!! 
		}
		//else if(!isnan(v) && !isnan(bounds[0]) && v == bounds[0])
		else if(node == bounds[0].value)
		{
			ret = my_get_count_rec(r, bounds, lenbounds, beg, n2); //have a look at the left side
		}
		//else if(!isnan(v) && !isnan(bounds[0]) && v > bounds[0])
		else if(node > bounds[0].value)
		{
			ret = my_get_count_rec(r, bounds, lenbounds, beg, n2);
		}
		return ret;
		
	}
	else
	{
		//if(!isnan(v) && !isnan(bounds[0]) && v < bounds[0])
		if(node > bounds[0].value)
		{
			ret = my_range_tree_get_count(r->list + i, bounds + 1, lenbounds - 1, bounds[0].is_positive); //go to the range tree of the following attribute
			ret += my_get_count_rec(r, bounds, lenbounds, beg, n2); //go to the left side!! 
		}
		//else if(!isnan(v) && !isnan(bounds[0]) && v == bounds[0])
		else if(node == bounds[0].value)
		{
			ret = my_get_count_rec(r, bounds, lenbounds, i + 1, n - n2 - 1); //have a look at the right side
		}
		//else if(!isnan(v) && !isnan(bounds[0]) && v > bounds[0])
		else if(node < bounds[0].value)
		{
			ret = my_get_count_rec(r, bounds, lenbounds, i + 1, n - n2 - 1);
		}
		return ret;
	}
	
}



//new
size_t
my_range_tree_get_count(range_tree * r, Bound * bounds, size_t lenbounds, unsigned short int is_positive) 
{
    if(lenbounds == 0)
	{
		if (is_positive != 0)
			return r->counter;
		else
			return r->counterGrt;
	}

	//my_print_rt(r);
    return my_get_count_rec(r, bounds, lenbounds, 0, r->listlen);
}


size_t
range_tree_get_count(range_tree * r, double * bounds, size_t lenbounds)
{
    if(lenbounds == 0)
		return r->counter;
    return get_count_rec(r, bounds, lenbounds, 0, r->listlen);
}


void
range_tree_remove_dimension(range_tree * r, size_t n)
{
    size_t i;
	
    if(r->list == NULL)
    {
		return;
    }
    if(n == 2)
    {
		range_tree_free_rec(r);
		r->list = NULL;
    }
    else
    {
		for(i = 0; i < r->listlen; i++)
		{
			range_tree_remove_dimension(r->list + i, n - 1);
		}
    }
}

