/*
 * Scan matrices for close invertible additive groups.
 *
 */

#ifndef ALPHABET64_H
#define ALPHABET64_H

#include "matrix.h"
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <vector>
#include <string>
#include <getopt.h>
#include <x86intrin.h>
#include <algorithm>

// =============================================================================
// Alphabet64 class
// =============================================================================

template <int m>
class Alphabet64 {
private:
    static constexpr int N = 1 << m;
    MatrixR64<m> t[N];                                                          // Tuple; we use a static array for efficiency
    int n;                                                                      // Number of valid elements
    inline bool findOrInsert(const MatrixR64<m> &X);                            // Try to insert a cnddt if it passes initial screening of an empty slot, return false on failure
public:
    Alphabet64(TupleR64<m> seed = {});                                            // Initialize to a given tuple, filtering invalid entries, first comes first served!
    static Alphabet64 getRandom(uint64_t maxAttempts = 0xFFFFFFFF);               // Construct a random alphabet, or give up after maxAttempts attempts
    bool complete(uint64_t maxAttempts = 0xFFFFFFFF);                           // Try to complete the alphabet by randomly searching for a primitive, try no more than maxAttempts times
    Alphabet64<2*m> upsample();                                                   // Generate a 2m alphabet that nests this
    void sortNest(int q0 = 1);                                                  // Sort alphabet to make it ready for nesting, q0 is atom bit resolution
    int getn() { return n; };
    inline bool operator+=(const MatrixR64<m> &cnddt);                          // Try to add cnddt to tuple, and return false if it fails
    inline Alphabet64 operator+(const MatrixR64<m> &cnddt) {
        Alphabet64 result(*this); result += cnddt; return result;
    }
    inline bool operator==(const Alphabet64<m> &other);                           // Equality test
    inline uint64_t nextSlot();                                                 // Return the next empty slot in most-significant row, or 2^(m * m) if all set
    operator bool() { return n == N; }                                          // Validity test
    operator TupleR64<m>() {                                                    // Export a TupleR64 for external use
        return TupleR64<m>(std::begin(t), std::begin(t) + n);
    }
    TupleR64<m> primitives() const;                                             // Return sorted list of primitives
    TupleR generators(int mOut = 64);                                           // Construct generator matrices from alphabet
    MatrixR getCap(int i, int mOut = 64);                                       // Returns the triangular cap of the ith generator
};

// -----------------------------------------------------------------------------
// Construct a random alphabet, try as much as 'attempts' times
// -----------------------------------------------------------------------------

template <int m>
Alphabet64<m> Alphabet64<m>::getRandom(uint64_t maxAttempts) {
    for (uint64_t attempt = 0; attempt < maxAttempts; attempt++) {
        Alphabet64 alphabet{{rnd()}};
        if (alphabet) {
            //fprintf(stderr, "Hit a primitive in %u attempts\n", attempt);       // Comment out as needed
            return alphabet;
        }
    }
    fprintf(stderr, "Failed to create the required alphabet; Aborting!\n");
    exit(1);
}

// -----------------------------------------------------------------------------
// Try to complete by randomly searching for a primitive
// -----------------------------------------------------------------------------

template <int m>
bool Alphabet64<m>::complete(uint64_t maxAttempts) {
    for (uint64_t attempt = 0; attempt < maxAttempts; attempt++) {
        Alphabet64 alphabet = *this + MatrixR64<m>(rnd());
        if (alphabet) {
            //fprintf(stderr, "Hit a primitive in %u attempts\n", attempt);       // Comment out as needed
            *this = alphabet;
            return true;
        }
    }
    fprintf(stderr, "Alphabet64 completion failed\n");                            // Comment out as needed
    return false;
}


// -----------------------------------------------------------------------------
// Try to insert a candidate if it passes initial screening of and empty slot
// -----------------------------------------------------------------------------

template <int m>
Alphabet64<m>::Alphabet64(TupleR64<m> seed): t({0, MatrixR64<m>::I()}), n(2) {      // Initialize to 0 and I, and mark their slots
    for (int i = 0; i < seed.size(); i++) {
        *this += seed[i];
    }
}

// -----------------------------------------------------------------------------
// Try to insert a candidate if it passes initial screening of and empty slot
// -----------------------------------------------------------------------------

template <int m>
bool Alphabet64<m>::findOrInsert(const MatrixR64<m> &X) {
    for (int i = 0; i < n; i++) {
        uint64_t sum = X + t[i];
        if (!sum) return true;                                                  // Found
        if (!(sum >> (m * (m - 1)))) return false;                              // Slot taken
    }
    // -------------------------------------------------------------------------
    // Not found and slot empty:
    // -------------------------------------------------------------------------
    t[n++] = X;
    return true;
}

// -----------------------------------------------------------------------------
// Try to add candidate to tuple, and return false if it fails
// -----------------------------------------------------------------------------

template <int m>
bool Alphabet64<m>::operator+=(const MatrixR64<m> &cnddt) {
    if ( n == N) {                                                              // Full occupacy, then cnddt must already be there to be accepted
        for (int i = 0; i < n; i++) {
            if (cnddt == t[i]) { return true; }
        }
        return false;
    }
    int nFallBack(n);
    bool pass = findOrInsert(cnddt);                                            // Place on top of list if not occupied
    for (int i = nFallBack; (i < n) && pass; i++) {                             // Iterate through newly added matrices to validate them
        MatrixR64<m> X = t[i];                                                  // Retrieve a matrix
        pass = (
            X.isInvertible()                                                    // Minimum requirement after screening, more tests apply to derivatives
            && findOrInsert(X.inverse())                                        // First derivative is invert itself
        );
        for (int j = 0; (j < i) && pass; j++) {                                 // Iterate through all preceding entries
            pass = (
                pass
                && findOrInsert(X + t[j])                                       // Sum to each preceding element
                && findOrInsert(X * t[j])                                       // Right multiplication by each preceding element
                && findOrInsert(t[i] * X)                                       // Left multiplication by each preceding element
            );
        }
    }
    if (!pass) {                                                                // If it fails at any step
        n = nFallBack;                                                          // Just ignore the inserted elements and return counter to initial set
    }
    return pass;
}

// -----------------------------------------------------------------------------
// Return sorted list of primitives
// -----------------------------------------------------------------------------

template <int m>
TupleR64<m> Alphabet64<m>::primitives() const {
    if (n != N) return {};                                                      // We only deal with complete alphabets
    if (N == 2) return {t[1]};                                                  // Identity is the primitive for alphabet<1>
    TupleR64<m> out;
    for (int i = 2; i < n; i++) {
        int minRootIndex(1);
        for (MatrixR64<m> Power = t[i]; Power != t[1]; Power = t[i] * Power) {
            minRootIndex++;
        }
        if (minRootIndex == N - 1) {
            out.push_back(t[i]);
        }
    }
    std::sort(out.begin(), out.end());
    return out;
}


// -----------------------------------------------------------------------------
// Try to insert a candidate if it passes initial screening of and empty slot
// -----------------------------------------------------------------------------

template <int m>
uint64_t Alphabet64<m>::nextSlot() {
    std::vector<bool> vacant(N, true);
    for (int i = 0; i < n; i++) {
        vacant[t[i] >> (m * (m - 1))] = false;
    }
    for (uint64_t i = 0; i < N; i++) {
        if (vacant[i]) {
            return i << (m * (m - 1));
        }
    }
    return N << (m * (m - 1));
}

// -----------------------------------------------------------------------------
// Equality test
// -----------------------------------------------------------------------------

template <int m>
bool Alphabet64<m>::operator==(const Alphabet64<m> &other) {
    if (n != other.n) {
        return false;
    }
    return primitives()[0] == other.primitives()[0];
}

// -----------------------------------------------------------------------------
// Generate a 2m alphabet that nests this
// -----------------------------------------------------------------------------

template <int m>
Alphabet64<2*m> Alphabet64<m>::upsample() {
    TupleR64<2*m> seed;
    for (int i = 0; i < n; i++) {
        MatrixR64<m> aSq = t[i] * t[i];
        seed.push_back(
            MatrixR64<2*m>{0}
            .putBlock(0, 0, m, aSq)
            .putBlock(m, m, m, aSq)
        );
    }
    return Alphabet64<2*m>{seed};
}


// -----------------------------------------------------------------------------
// Sort to make ready for nesting
// -----------------------------------------------------------------------------

template <int m>
void Alphabet64<m>::sortNest(int q0) {
    if (n != N) { return; }                                                     // Sorry, we only accept complete alphabets
    for (int q = q0; q < m; q++) {
        int nElements = 1 << q;
        MatrixR64<m> keyElement = t[nElements];                                 // First element in second band
        for (int i = 1; i < nElements; i++) {                                   // Iterate through elements of first band
            int slotIndex = i + nElements;                                      // Slots in the second band
            MatrixR64<m> slotOwner = keyElement + t[i];                         // Slot should be occupied by this element
            for (int j = slotIndex; j < n; j++) {                               // Scan the slots
                if (t[j] == slotOwner) {
                    std::swap(t[j], t[slotIndex]);
                    break;
                }
            }
        }
    }
}


// -----------------------------------------------------------------------------
// Construct generator matrices from alphabet
// -----------------------------------------------------------------------------

template <int m>
TupleR Alphabet64<m>::generators(int mOut){
    TupleR out;
    for (int i = 0; i < n; i++) {                                               // Iterate through symbols
        MatrixR L = MatrixR::I(mOut).shift(m, 0) + MatrixR::IBlock(mOut, t[i]); // Lower-triangular for recurrence formula
        MatrixR col = MatrixR(mOut).putBlock(0, 0, t[1]);                       // Identity on first element
        MatrixR C(mOut);                                                        // Start with an empty matrix
        for (int j = 0; j < mOut; j += m) {
            C = C + col.shift(0, j);
            col = L * col;
        }
        out.push_back(C);
    }
    return out;
}

// -----------------------------------------------------------------------------
//
// -----------------------------------------------------------------------------

template <int m>
MatrixR Alphabet64<m>::getCap(int i, int mOut){
    return MatrixR::IBlock(mOut, MatrixR::I(2*m).putBlock(0, m, t[i]));
}

// =============================================================================
// Print arithmetic table
// =============================================================================

template <int m>
void printTable(
    TupleR64<m> t,
    int tableType = 0,                                                          // 0: addition, 1: multiplication, 2: inversion
    FILE *file = stdout
) {
    const char *title[] = {"Addition:", "Multiplication:", "Inversion:"};
    //printf("%s:\n", title[tableType]);
    int n = t.size();
    MatrixR64<m>::printf(t, file, title[tableType], 6);
    std::vector<int> table(n * n);
    for (int i = 0; i < n; i++) {
        TupleR64<m> row;
        row.push_back(t[i]);                                                    // Key column
        for (int j = 0; j < n; j++) {
            MatrixR64<m> result;
            switch(tableType) {
                case 0: result = t[i] + t[j]; break;
                case 1: result = t[i] * t[j]; break;
                case 2: result = (t[i] * t[j] == t[1] ? t[1] : t[0]); break;    // Assuming t[0] is 0 and t[1] is I
            }
            row.push_back(result);                                              // Insert actual matrix in row
            table[i * n + j] = findOrAbort(result, t);                          // Insert id in sympolic table
        }
        MatrixR64<m>::printf(row, file, "\n");                                  // Display resultant row
    }
    fprintf(file, "\nSymbolic Table:\n  ");
    for (int i = 0; i < n; i++) printf("%3X", i);
    fprintf(file, "\n");
    for (int i = 0; i < n; i++) {
        fprintf(file, "%2X", i);
        for (int j = 0; j < n; j++) {
            fprintf(file, "%3X", table[i * n + j]);
        }
        fprintf(file, "\n");
    }
    fprintf(file, "\n\n");
}



#endif                                                                          // #ifndef ALPHABET64_H
