/*
 * Commonly used elements, like random number generator
 */

#ifndef COMMON_H
#define COMMON_H

#include <random>
#include <string>

#ifdef USE_DOUBLE
    #define Float double
    #define twoExpMinus32 2.3283064365386962890625e-10
#else
    #define Float float
    #define twoExpMinus32 2.3283064365386962890625e-10f
#endif


// =============================================================================
// Data types
// =============================================================================

struct Point {
    uint32_t x, y;
};

typedef std::vector<Point>          Points;
typedef std::string                 String;
typedef std::vector<int>            List;



// =============================================================================
// Common parameters
// =============================================================================

const char SYMBOLS[] = {'.', '1', ' '};                                         // For printing binary matrices; third is useful for some scenarios

// =============================================================================
// Utils
// =============================================================================

std::random_device rd;                                                          // Will be used to obtain a seed for the random number engine
std::mt19937_64 rng(rd());
#define rnd() static_cast<uint64_t>(rng())

inline uint64_t pow2(int power) { return 1llu << power; }                       // 2^m

inline uint64_t msk(int k) { return 0xFFFFFFFFFFFFFFFFllu >> (64 - k); }        // We used "pow2(k) - 1" fist, but that would not reach 64 bits

// -----------------------------------------------------------------------------
// Fixed point integer representation to floating point
// -----------------------------------------------------------------------------

inline Float fxd2flt(uint32_t x) {
    return twoExpMinus32 * x;
}

// -----------------------------------------------------------------------------
// Searching for an entry in a list; we use simple linear search, assuming
// relatively short lists
// -----------------------------------------------------------------------------

template <typename T>
inline int findOrInsert(T entry, std::vector<T> &list) {                        // This would add an item if not found
    for (int index = 0; index < list.size(); index++) {
        if (list[index] == entry) return index;
    }
    list.push_back(entry);
    return list.size() - 1;
}

template <typename T>
inline int findOrAbort(T entry, std::vector<T> &list) {                         // This will abort execution if item is not found
    for (int index = 0; index < list.size(); index++) {
        if (list[index] == entry) return index;
    }
    fprintf(stderr, "ERROR: element not found! Aborting\n");
    exit(1);
}

// -----------------------------------------------------------------------------
// Shuffle entries of a given list
// -----------------------------------------------------------------------------

template <typename T>
inline void shuffle(std::vector<T> &list) {
    uint32_t N = list.size();
    for (uint32_t i = N - 1; i > 0; i--) {                                      // Iterate down the list, and swap each place with another place down, inclusive of same place.
        uint32_t r = rnd() % (i + 1);                                           // (i + 1) means that the same place is included.
        std::swap(list[i], list[r]);
    }
}

// -----------------------------------------------------------------------------
// Print fixed-point points to a txt file
// -----------------------------------------------------------------------------

void printTxt(const Points &p, const char *fileName) {
    FILE *file = fopen(fileName, "w");
    fprintf(file, "%d\n", p.size());
    constexpr double res = double(1) / (1llu << 32);
    for (int i = 0; i < p.size(); i++) {
        fprintf(file, "%0.17f %0.17f\n", res * p[i].x, res * p[i].y);
    }
    fclose(file);
}

// -----------------------------------------------------------------------------
// Owen-scramble a list of points
// -----------------------------------------------------------------------------

void owenScramble(Points &p) {
    int N = p.size();
    int m = __builtin_ctz(N);
    uint32_t *x = (uint32_t *)p.data();                                         // Cast to uint32_t array so that we can loop
    for (int dim = 0; dim < 2; dim++) {
        std::vector<uint32_t> map(N);
        for (int i = 0; i < N; i++) { map[i] = i; }
        for (int bitPosition = 0; bitPosition < m; bitPosition++) {             // 0 is root, m - 1 is leaves level
            int branchSize = 1 << (m - bitPosition);
            int flag;
            for (int i = 0; i < N; i++) {
                if ((i & (branchSize - 1)) == 0) {                              // Beginning of new branch
                    flag = (rnd() & 1) << (m - 1 - bitPosition);
                }
                map[i] ^= flag;
            }
        }
        for (int i = dim; i < 2 * N; i += 2) {
            x[i] = (
                (map[x[i] >> (32 - m)] << (32 - m)) |                       // Map significant bits to a new interval
                (rnd() & msk(32 - m))                                       // Random trailing bits
            );
        }
    }
}

void owenScramble(uint32_t *x, int N, int stride = 1) {
    int m = __builtin_ctz(N);
    std::vector<uint32_t> map(N);
    for (int i = 0; i < N; i++) { map[i] = i; }
    for (int bitPosition = 0; bitPosition < m; bitPosition++) {                 // 0 is root, m - 1 is leaves level
        int branchSize = 1 << (m - bitPosition);
        int flag;
        for (int i = 0; i < N; i++) {
            if ((i & (branchSize - 1)) == 0) {                                  // Beginning of new branch
                flag = (rnd() & 1) << (m - 1 - bitPosition);
            }
            map[i] ^= flag;
        }
    }
    for (int i = 0; i < stride * N; i += stride) {
        x[i] = (
            (map[x[i] >> (32 - m)] << (32 - m)) |                               // Map significant bits to a new interval
            (rnd() & msk(32 - m))                                               // Random trailing bits
        );
    }
}

void xorScramble(Points &p) {
    int N = p.size();
    int m = __builtin_ctz(N);
    uint32_t *x = (uint32_t *)p.data();                                         // Cast to uint32_t array so that we can loop
    for (int dim = 0; dim < 2; dim++) {
        uint32_t scramble = static_cast<uint32_t>(rnd());
        for (int i = dim; i < 2 * N; i += 2) {
            x[i] ^= scramble;
        }
    }
}

// =============================================================================
// A counter class to iterate through combinations
// =============================================================================

class CombinationCounter {
private:
    std::vector<int> digits;
    const uint64_t indexMax;
    const int nSlots;                                                           // Positional places, = number of loops
    bool resetFlag;                                                             // Flag raised when the counter is full and reset
    inline int digitMax(int i) { return indexMax - (nSlots - 1 - i); };
    bool inc(int i) {                                                           // Increment the ith digit and return carry, reset if full
        if (digits[i] == digitMax(i) ) { digits[i] = 0; return true; }          // If digit is full, zero it and return a carry
        /*else*/                       { digits[i]++;   return false; }         // Otherwise, just increment digit, no carry
    }
    inline void calibrate();
public:
    CombinationCounter(int nElements, int nSlots):
        indexMax(nElements - 1),
        nSlots(nSlots),
        digits(nSlots, 0),
        resetFlag(false)
        { calibrate(); };                                                       // Construct a counter and reset it
    void operator++();                                                          // Increment the counter, set resetFlag if reset
    void operator++(int) { ++(*this); };
    std::vector<int> getDigits() { return digits; };
    operator bool() { return !resetFlag; }                                      // Cast into bool indicating more to count
};

// -----------------------------------------------------------------------------

void CombinationCounter::operator++() {
    bool carry(true);                                                           // The added increment on the right, potentially propagating
    for (int i = nSlots - 1; carry && (i >= 0); i--) {
        carry = inc(i);
    }
    if (carry) { resetFlag = true; return; }                                    // The counter is full and reset
    calibrate();
}

void CombinationCounter::calibrate() {
    for (int i = 1; i < nSlots; i++) {
        if (digits[i] <= digits[i - 1]) { digits[i] = digits[i - 1] + 1; }      // Enforce the key property that less-significant digits must be more than leading ones
    }
}


// =============================================================================
// A counter class to iterate through combinations
// =============================================================================

class CombinationCounter64 {
private:
    uint64_t digits;                                                            // Actual contents
    const uint64_t indexMax;
    const int nSlots;                                                           // Positional places, = number of loops
    bool resetFlag;                                                             // Flag raised when the counter is full and reset
    uint64_t digit(int i) { return (digits >> (i * BITs)) & MASK; };            // Retrieve the ith digit
    void set(int i, uint64_t v) { digits ^= (digit(i) ^ v) << (i * BITs); }     // Set value of ith digit
    bool inc(int i);                                                            // Increment the ith digit and return carry, reset if full
    inline void calibrate();
public:
    const int BITs = 4;                                                         // We use 4 bits per digit, so max is 16 elements and 16 slots
    const uint64_t MASK = msk(BITs);
    CombinationCounter64(int nElements, int nSlots):
        indexMax(nElements - 1),
        nSlots(nSlots),
        digits(0),
        resetFlag(false)
        { calibrate(); };                                                       // Construct a counter and reset it
    void operator++();                                                          // Increment the counter, set resetFlag if reset
    void operator++(int) { ++(*this); };
    uint64_t getDigits() { return digits; };
    operator bool() { return !resetFlag; }                                      // Cast into bool indicating more to count
};

// -----------------------------------------------------------------------------

bool CombinationCounter64::inc(int i) {
    if (digit(i) == indexMax - i) {                                                  // Reached maximum for ith digit?
        digits ^= digits & (MASK << (i * BITs));                                // Zero it
        return true;                                                            // Return a carry
    }
    digits += 1llu << (i * BITs); return false;                                 // Otherwise, just increment digit, no carry
}

// -----------------------------------------------------------------------------

void CombinationCounter64::operator++() {
    bool carry(true);                                                           // The added increment on the right, potentially propagating
    for (int i = 0; carry && (i < nSlots); i++) {
        carry = inc(i);
    }
    if (digit(nSlots - 1) >= indexMax) { resetFlag = true; }
    if (carry) { resetFlag = true; return; }                                    // The counter is full and reset
    calibrate();
}

void CombinationCounter64::calibrate() {
    for (int i = nSlots - 1; i > 0; i--) {
        if (digit(i - 1) <= digit(i)) { set(i - 1, digit(i) + 1); }             // Enforce the key property that less-significant digits must be more than leading ones
    }
}


// =============================================================================
// A counter class to handle nested loops in building hybrid matrices
// =============================================================================

class NestedCounter64 {
private:
    uint64_t digits;                                                            // Actual contents
    const uint64_t max;                                                         // = max in each loop; declared as uint64_t because it is gonna be shifted; a subtle bug!
    const int digitCount;                                                       // Positional places, = number of loops
    bool resetFlag;                                                             // Flag raised when the counter is full and reset
    uint64_t digit(int i) { return (digits >> (i * BITs)) & MASK; };            // Retrieve the ith digit
    void set(int i, uint64_t v) { digits ^= (digit(i) ^ v) << (i * BITs); }     // Set value of ith digit
    int inc(int i) {                                                            // Increment the ith digit and return carry, reset if full
        if (digit(i) == max) { digits -= max  << (i * BITs); return 1; }        // If digit is full, zero it and return a carry of 1
        /*else*/             { digits += 1llu << (i * BITs); return 0; }        // Otherwise, just increment digit, no carry
    }
public:
    const int BITs = 4;                                                         // We use 4 bits per digit, so max digitCount = 16 (64/4) and max base 16 (2^4)
    const uint64_t MASK = msk(BITs);
    NestedCounter64(int max, int digitCount):
        max(max),
        digitCount(digitCount),
        digits(0),
        resetFlag(false)
        {};                                                                     // Construct a counter and reset it
    void operator++();                                                          // Increment the counter, set resetFlag if reset
    void operator++(int) { ++(*this); };
    uint64_t getDigits() { return digits; };
    uint64_t getCounts() { return (digits << BITs) + max - digits; };               // How many to take from each to sum to base, including a "last" other element
    operator bool() { return !resetFlag; }                                      // Cast into bool indicating more to count
};

// -----------------------------------------------------------------------------

void NestedCounter64::operator++() {
    int carry(1);                                                               // The added increment on the right, potentially propagating
    for (int i = 0; carry && (i < digitCount); i++) {
        carry = inc(i);
    }
    if (carry) { resetFlag = true; return; }                                    // The counter is full and reset
    for (int i = digitCount - 1; i > 0; i--) {
        if (digit(i - 1) < digit(i)) { set(i - 1, digit(i)); }                  // Enforce the key property that less-significant digits may not be less than leading ones
    }
}

// =============================================================================
// A nested counter using arrays to deal with more than 16 digits
// =============================================================================

class NestedCounter {
private:
    std::vector<int> digits;                                                    // Actual contents
    const uint64_t max;                                                         // = max in each loop; declared as uint64_t because it is gonna be shifted; a subtle bug!
    const int digitCount;                                                       // Positional places, = number of loops
    bool resetFlag;                                                             // Flag raised when the counter is full and reset
    bool inc(int i) {                                                           // Increment the ith digit and return carry, reset if full
        if (digits[i] == max) { digits[i] = 0; return true; }                   // If digit is full, zero it and return a carry
        /*else*/              { digits[i]++;   return false; }                  // Otherwise, just increment digit, no carry
    }
public:
    NestedCounter(int max, int digitCount):
        max(max),
        digitCount(digitCount),
        digits(digitCount, 0),
        resetFlag(false)
        {};                                                                     // Construct a counter and reset it
    void operator++();                                                          // Increment the counter, set resetFlag if reset
    void operator++(int) { ++(*this); };
    std::vector<int> getDigits() { return digits; };
    std::vector<int> getCounts();                                               // How many to take from each to sum to base, including a implicit "last" other element
    operator bool() { return !resetFlag; }                                      // Cast into bool indicating more to count
};

// -----------------------------------------------------------------------------

void NestedCounter::operator++() {
    bool carry(true);                                                           // The added increment on the right, potentially propagating
    for (int i = 0; carry && (i < digitCount); i++) {
        carry = inc(i);
    }
    if (carry) { resetFlag = true; return; }                                    // The counter is full and reset
    for (int i = digitCount - 1; i > 0; i--) {
        if (digits[i - 1] < digits[i]) { digits[i - 1] = digits[i]; }          // Enforce the key property that less-significant digits may not be less than leading ones
    }
}

std::vector<int> NestedCounter::getCounts() {
    if (!digitCount) {
        return std::vector<int>(1, max);
    }
    std::vector<int> counts(digitCount + 1, 0);
    counts[0] = max - digits[0];                                                // digits[-1] = max
    counts[digitCount] = digits[digitCount - 1];                                // digits[digitCount] = 0
    for (int i = 1; i < digitCount; i++) {
        counts[i] = digits[i - 1] -  digits[i];
    }
    return counts;
}

#endif                                                                          // #ifndef COMMON_H
