/*
 * Miscellaneous tests on sz sequences
 */

#ifndef SZ_H
#define SZ_H

#include "common.h"
#include "alphabet.h"
#include "alphabet64.h"
#include <vector>
#include <string>
#include <getopt.h>

const int BITs = 32;
extern const uint32_t SobolColumnMatrices[16*32];

// =============================================================================
// Base sequence class
// =============================================================================

class Sequence {
protected:
    int dims;                                                                   // Dimensions
    TupleR C;                                                                   // Generator matrices
public:
    Sequence(): dims(0), C({}) {};                                              // Construct an empty sequence
    Sequence(const TupleR &C): C(C), dims(C.size()) {};                         // Construct from a given tuple of row matrices
    Sequence(int dims, const uint32_t *cols);                                   // Convert from a list of column matrices like PBRT's
    int getDims() const { return dims; }
    void printf(FILE *file = stdout, const char *txt = NULL);
    void printC32(FILE *file = stdout, const char *txt = NULL);
    operator TupleR() const { return C; };
    const MatrixR& operator[](int dim) const { return C[dim]; };                // Retrieve a constituent matrix
    uint32_t operator()(int seqNo, int dim) const { return C[dim][seqNo]; };    // Retrieve a sample point in a given dimension
    std::vector<List> netMatches(bool withRordering = false);                   // Enumerate empirically netMatched subsets of dimensions
    void scanEnsembles(                                                         // Scan for ensembled(0, s)-sequences
        int q0 = 1, FILE *file = stdout, bool withRordering = true
    );
    static Sequence Sobol();                                                    // Build a 16D Sobol sequence from PBRT tables
};

// -----------------------------------------------------------------------------
// Convert from a list of column matrices like PBRT's
// -----------------------------------------------------------------------------

Sequence::Sequence(int dims, const uint32_t *cols): dims(dims), C(dims, 32) {
    for (int dim = 0; dim < dims; dim++) {
        C[dim] = MatrixR(cols + dim * 32, 32, true);
    }
}


// -----------------------------------------------------------------------------
// Print matrices in binary
// -----------------------------------------------------------------------------

void Sequence::printf(FILE *file, const char *txt) {
    MatrixR::printf(C, file, txt);
}

// -----------------------------------------------------------------------------
// Print matrices in C format
// -----------------------------------------------------------------------------

void Sequence::printC32(FILE *file, const char *txt) {
    if (txt) fprintf(file, "%s\n", txt);
    for (int dim = 0; dim < dims; dim++) {
        C[dim].printC32(file);
        fprintf(file, "\n");
    }
}


// -----------------------------------------------------------------------------
// Scan for netmatching, mainly used for Sobol, but is not for general
// use due to the fast growing number of combinations
// -----------------------------------------------------------------------------

std::vector<List> Sequence::netMatches(bool withRordering) {
    for (int q = 1, s = 2; q <= 4 && s <= dims; q++, s <<= 1) {
    //for (int q = 4, s = 16; q <= 4 && s <= dims; q++, s <<= 1) {
        int bits = q * (BITs / q);                                              // Round down to nearest multiple of q
        CombinationCounter cntr(dims, s);
        for (; cntr; cntr++) {
            std::vector<int> dimIndex = cntr.getDigits();
            ::printf("Testing");
            for (int i = 0; i < s; i++) {
                ::printf(" %2x", dimIndex[i]);
            }
            ::printf("\n");
            TupleR t;
            String txt = "C[";
            for (int i = 0; i < s; i++) {
                int dim = dimIndex[i];
                t.push_back(C[dim].setm(bits));
                txt += std::to_string(dim);
                if (i < s - 1) { txt += ", "; }
            }
            txt += "] make a (0, " + std::to_string(s) + ")-sequence\n";
            if (withRordering) {
                MatrixR order = t[0].inverse();
                for (int i = 0; i < s; i++) {
                    t[i] = t[i] * order;
                }
            }
            bool pass = true;
            for (int m = q; m < BITs && pass; m += q) {                         // Iterate through net sizes
                TupleR leading = MatrixR::setm(t, m);
                pass = MatrixR::netMatch(leading, q);
            }
            if (pass) {
                fprintf(stdout, txt.c_str());
            }
        }
    }
    return {};
}

// -----------------------------------------------------------------------------
// Scan for ensembled (0, s)-sequences
// -----------------------------------------------------------------------------

void Sequence::scanEnsembles(int q0, FILE *file, bool withRordering) {
    for (int q = q0, s = (1 << q); s <= dims; q *= 2, s *= s) {                 // Iterate through sub-sequence dimension
        for (int i0 = 0; i0 < dims; i0 += s) {                                  // Iterate through bands of sub-sequences
            TupleR t;
            fprintf(file, "(");
            for (int i = 0; i < s; i++) {
                t.push_back(C[i + i0]);
                fprintf(file, "%2x%s", i + i0, i < s - 1 ? "," : "): ");
            }
            if (MatrixR::seqMatch(t, q, withRordering)) {
                fprintf(file, "==========");                                    // We use ========== as an easy to see pattern to indicate (0, s)-sequence
            }
            fprintf(file, "\n");
        }
    }
}


// -----------------------------------------------------------------------------
// PBRT Sobol for comparison
// -----------------------------------------------------------------------------

Sequence Sequence::Sobol() {
    TupleR C(16);
    for (int dim = 0; dim < 16; dim++) {
        std::vector<uint64_t> R(32, 0);
        const uint32_t *cols = SobolColumnMatrices + 32 * dim;
        for (int i = 0; i < 32; i++) {
            for (int j = 0; j < 32; j++) {
                R[i] |= ((cols[j] >> (31 - i)) & 1) << j;
            }
        }
        C[dim] = R;
    }
    return Sequence(C);
}

// =============================================================================
// SZ sequence class
// =============================================================================

class SZ : public Sequence {
protected:
    int q;
public:
    SZ(): Sequence(), q(0) {};
    SZ(int q, bool randomize = true, bool validate = false);
    SZ(const Sequence &seq);                                                    // Try to cast a given sequence into an SZ, if it is.
    static SZ makeNested(
        int q0 = 1,
        int levels = 2,
        bool ensemble = true,
        bool randomize = true
    );
    Alphabet getAlphabet() const;                                               // Extract alphabet from generators
    SZ upsample(bool ensemble = true, bool randomize = true) const;             // Return a sequence that is an upsampled version of this, leaving this intact
};

// -----------------------------------------------------------------------------
// Create a random SZ sequence
// -----------------------------------------------------------------------------

SZ::SZ(int q, bool randomize, bool validate): q(q) {
    dims = 1 << q;
    int bits = q * ((BITs + q - 1) / q);                                        // Round up to nearest multiple of q
    C = Alphabet::getRandom(q).sortAlpha().generators(bits);
    if (randomize) {
        for (int dim = 1; dim < dims; dim++) {
            MatrixR s = MatrixR::I(bits);
            for (int i = 0; i < bits; i += q) {
                s = s.putBlock(i, i, MatrixR::createInvertible(q));
                //s = s.putBlock(i, i, MatrixR::U(q));
            }
            C[dim] = s * C[dim];
            //C[dim] = MatrixR::L(bits) * C[dim];

        }
    }
    if (validate) {
        if (MatrixR::seqMatch(C, q, false)) {
            fprintf(stderr, "Created and validated an SZ_%d sequence\n", q);
        }
        else {
            fprintf(stderr, "Validation failed for SZ_%d, Aborting\n", q);
            exit(1);
        }
    }
    for (int dim = 0; dim < dims; dim++) {
        C[dim] = C[dim].setm(BITs);                                             // Trim to used bit resolution
    }
}

// -----------------------------------------------------------------------------
// Try to cast a given sequence into an SZ, if it is.
// -----------------------------------------------------------------------------

SZ::SZ(const Sequence &seq): Sequence(seq) {                                    // Initialize to seq, in hope it works
    q = __builtin_ctz(dims);
    if (!getAlphabet()) {                                                       // Invalid alphabet?
        C = {}; dims = 0; q = 0;                                                // Return an empty sequence
    }
}

// -----------------------------------------------------------------------------
// Created a nested SZ sequence
// -----------------------------------------------------------------------------

SZ SZ::makeNested(int q0, int levels, bool ensemble, bool randomize) {
    int dims = 1 << (q0 << levels);
    int imax = ensemble ? dims : 1;                                             // Curl all or only first band?
    Alphabet a{q0};
    a.complete();
    a.sortNest();
    std::vector<TupleR> caps;
    for (int level = 0; level < levels; level++) {
        int q = q0 << level;
        TupleR c;
        for (int i = 0; i < (1 << q); i++) {
            c.push_back(a.getCap(i, BITs));
        }
        caps.push_back(c);
        a = a.upsample();
    }
    TupleR C = a.generators(BITs);
    if (randomize) {
        // Curl with own alphabet to decorrelate
        TupleR s = a;
        for (int level = 0; level < levels; level++) {
            int q = q0 << level, nElements = 1 << q;
            for (int i0 = nElements; i0 < dims; i0 += nElements) {
                int symbolIndex = 1 + rnd() % (nElements - 1);
                MatrixR curl = MatrixR::IBlock(BITs, s[symbolIndex]);
                for (int i = 0; i < nElements; i++) {
                    C[i0 + i] = curl * C[i0 + i];
                }
            }
        }
        if (!ensemble) {                                                                  // Curl indepenetly with random blocks of nested sequence block size
            int q = q0 << levels;
            int nNested = 1 << (q/2);                                           // Number of nested dims
            for (int dim = nNested; dim < dims; dim++) {
                C[dim] = MatrixR::curl(BITs, q) * C[dim];                       // A random scurling
            }
        }
    }
    // -------------------------------------------------------------------------
    // s-curl with nested alphabets to restore lower-dimensional generators
    // -------------------------------------------------------------------------
    for (int level = levels - 1; level >= 0; level--) {
        int q = q0 << level, nElements = 1 << q;
        for (int i0 = 0; i0 < imax; i0 += nElements) {
            for (int i = 0; i < nElements; i++) {
                C[i0 + i] = caps[level][i] * C[i0 + i];
            }
        }
    }
    SZ out;
    out.dims = dims;
    out.C = C;
    out.q = q0 << levels;
    return out;
}

// -----------------------------------------------------------------------------
// Return an upsampled sequence
// -----------------------------------------------------------------------------

SZ SZ::upsample(bool ensemble, bool randomize) const {
    SZ out;
    out.dims = dims * dims;
    out.q = 2 * q;
    int q0 = this->q;                                                           // Just start with this
    int levels = 1;
    int imax = ensemble ? dims : 1;                                             // Curl all or only first band?
    Alphabet a = getAlphabet().upsample();
    TupleR g = a.generators(BITs);

    if (randomize) {
        // ---------------------------------------------------------------------
        // Curl bands with own alphabet to decorrelate
        // ---------------------------------------------------------------------
        TupleR s = a;
        for (int i0 = dims; i0 < out.dims; i0 += dims) {
            int symbolIndex = 1 + rnd() % (dims - 1);
            MatrixR curl = MatrixR::IBlock(BITs, s[symbolIndex]);
            for (int i = 0; i < dims; i++) {
                g[i0 + i] = curl * g[i0 + i];
            }
        }
        if (!ensemble) {                                                        // Curl indepenetly with random blocks of nested sequence block size
            // -----------------------------------------------------------------
            // Curl new dims with random matrices
            // -----------------------------------------------------------------
            for (int dim = dims; dim < out.dims; dim++) {
                g[dim] = MatrixR::curl(BITs, q) * g[dim];
            }
        }
    }
    // -------------------------------------------------------------------------
    // s-curl with nested alphabets to restore lower-dimensional generators
    // -------------------------------------------------------------------------
    MatrixR CMask = MatrixR::IBlock(BITs, MatrixR::repeatRow(2*q, msk(2*q)));
    for (int i = 0; i < dims; i++) {
        MatrixR c = C[i] & CMask;
        for (int j = i; j < (ensemble ? out.dims : dims); j += dims) {
            g[j] = c * g[j];
        }
    }
    out.C = g;
    return out;
}

// -----------------------------------------------------------------------------
// Try to cast a given sequence into an SZ, if it is.
// -----------------------------------------------------------------------------

Alphabet SZ::getAlphabet() const {
    TupleR x(dims);
    for (int dim = 0; dim < dims; dim++) {
        x[dim] = C[dim].getBlock(0, 0, q).inverse() * C[dim].getBlock(0, q, q);
    }
    Alphabet a{q, x};
    return a;
}



// =============================================================================
// Sobol Matrices
// =============================================================================

const uint32_t SobolColumnMatrices[16*32] = {
    0x80000000, 0x40000000, 0x20000000, 0x10000000,
    0x08000000, 0x04000000, 0x02000000, 0x01000000,
    0x00800000, 0x00400000, 0x00200000, 0x00100000,
    0x00080000, 0x00040000, 0x00020000, 0x00010000,
    0x00008000, 0x00004000, 0x00002000, 0x00001000,
    0x00000800, 0x00000400, 0x00000200, 0x00000100,
    0x00000080, 0x00000040, 0x00000020, 0x00000010,
    0x00000008, 0x00000004, 0x00000002, 0x00000001,

    0x80000000, 0xc0000000, 0xa0000000, 0xf0000000,
    0x88000000, 0xcc000000, 0xaa000000, 0xff000000,
    0x80800000, 0xc0c00000, 0xa0a00000, 0xf0f00000,
    0x88880000, 0xcccc0000, 0xaaaa0000, 0xffff0000,
    0x80008000, 0xc000c000, 0xa000a000, 0xf000f000,
    0x88008800, 0xcc00cc00, 0xaa00aa00, 0xff00ff00,
    0x80808080, 0xc0c0c0c0, 0xa0a0a0a0, 0xf0f0f0f0,
    0x88888888, 0xcccccccc, 0xaaaaaaaa, 0xffffffff,

    0x80000000, 0xc0000000, 0x60000000, 0x90000000,
    0xe8000000, 0x5c000000, 0x8e000000, 0xc5000000,
    0x68800000, 0x9cc00000, 0xee600000, 0x55900000,
    0x80680000, 0xc09c0000, 0x60ee0000, 0x90550000,
    0xe8808000, 0x5cc0c000, 0x8e606000, 0xc5909000,
    0x6868e800, 0x9c9c5c00, 0xeeee8e00, 0x5555c500,
    0x8000e880, 0xc0005cc0, 0x60008e60, 0x9000c590,
    0xe8006868, 0x5c009c9c, 0x8e00eeee, 0xc5005555,

    0x80000000, 0xc0000000, 0x20000000, 0x50000000,
    0xf8000000, 0x74000000, 0xa2000000, 0x93000000,
    0xd8800000, 0x25400000, 0x59e00000, 0xe6d00000,
    0x78080000, 0xb40c0000, 0x82020000, 0xc3050000,
    0x208f8000, 0x51474000, 0xfbea2000, 0x75d93000,
    0xa0858800, 0x914e5400, 0xdbe79e00, 0x25db6d00,
    0x58800080, 0xe54000c0, 0x79e00020, 0xb6d00050,
    0x800800f8, 0xc00c0074, 0x200200a2, 0x50050093,

    0x80000000, 0x40000000, 0x20000000, 0xb0000000,
    0xf8000000, 0xdc000000, 0x7a000000, 0x9d000000,
    0x5a800000, 0x2fc00000, 0xa1600000, 0xf0b00000,
    0xda880000, 0x6fc40000, 0x81620000, 0x40bb0000,
    0x22878000, 0xb3c9c000, 0xfb65a000, 0xddb2d000,
    0x78022800, 0x9c0b3c00, 0x5a0fb600, 0x2d0ddb00,
    0xa2878080, 0xf3c9c040, 0xdb65a020, 0x6db2d0b0,
    0x800228f8, 0x400b3cdc, 0x200fb67a, 0xb00ddb9d,

    0x80000000, 0x40000000, 0x60000000, 0x30000000,
    0xc8000000, 0x24000000, 0x56000000, 0xfb000000,
    0xe0800000, 0x70400000, 0xa8600000, 0x14300000,
    0x9ec80000, 0xdf240000, 0xb6d60000, 0x8bbb0000,
    0x48008000, 0x64004000, 0x36006000, 0xcb003000,
    0x2880c800, 0x54402400, 0xfe605600, 0xef30fb00,
    0x7e48e080, 0xaf647040, 0x1eb6a860, 0x9f8b1430,
    0xd6c81ec8, 0xbb249f24, 0x80d6d6d6, 0x40bbbbbb,

    0x80000000, 0xc0000000, 0xa0000000, 0xd0000000,
    0x58000000, 0x94000000, 0x3e000000, 0xe3000000,
    0xbe800000, 0x23c00000, 0x1e200000, 0xf3100000,
    0x46780000, 0x67840000, 0x78460000, 0x84670000,
    0xc6788000, 0xa784c000, 0xd846a000, 0x5467d000,
    0x9e78d800, 0x33845400, 0xe6469e00, 0xb7673300,
    0x20f86680, 0x104477c0, 0xf8668020, 0x4477c010,
    0x668020f8, 0x77c01044, 0x8020f866, 0xc0104477,

    0x80000000, 0x40000000, 0xa0000000, 0x50000000,
    0x88000000, 0x24000000, 0x12000000, 0x2d000000,
    0x76800000, 0x9e400000, 0x08200000, 0x64100000,
    0xb2280000, 0x7d140000, 0xfea20000, 0xba490000,
    0x1a248000, 0x491b4000, 0xc4b5a000, 0xe3739000,
    0xf6800800, 0xde400400, 0xa8200a00, 0x34100500,
    0x3a280880, 0x59140240, 0xeca20120, 0x974902d0,
    0x6ca48768, 0xd75b49e4, 0xcc95a082, 0x87639641,

    0x80000000, 0x40000000, 0xa0000000, 0x50000000,
    0x28000000, 0xd4000000, 0x6a000000, 0x71000000,
    0x38800000, 0x58400000, 0xea200000, 0x31100000,
    0x98a80000, 0x08540000, 0xc22a0000, 0xe5250000,
    0xf2b28000, 0x79484000, 0xfaa42000, 0xbd731000,
    0x18a80800, 0x48540400, 0x622a0a00, 0xb5250500,
    0xdab28280, 0xad484d40, 0x90a426a0, 0xcc731710,
    0x20280b88, 0x10140184, 0x880a04a2, 0x84350611,

    0x80000000, 0x40000000, 0xe0000000, 0xb0000000,
    0x98000000, 0x94000000, 0x8a000000, 0x5b000000,
    0x33800000, 0xd9c00000, 0x72200000, 0x3f100000,
    0xc1b80000, 0xa6ec0000, 0x53860000, 0x29f50000,
    0x0a3a8000, 0x1b2ac000, 0xd392e000, 0x69ff7000,
    0xea380800, 0xab2c0400, 0x4ba60e00, 0xfde50b00,
    0x60028980, 0xf006c940, 0x7834e8a0, 0x241a75b0,
    0x123a8b38, 0xcf2ac99c, 0xb992e922, 0x82ff78f1,

    0x80000000, 0x40000000, 0xa0000000, 0x10000000,
    0x08000000, 0x6c000000, 0x9e000000, 0x23000000,
    0x57800000, 0xadc00000, 0x7fa00000, 0x91d00000,
    0x49880000, 0xced40000, 0x880a0000, 0x2c0f0000,
    0x3e0d8000, 0x3317c000, 0x5fb06000, 0xc1f8b000,
    0xe18d8800, 0xb2d7c400, 0x1e106a00, 0x6328b100,
    0xf7858880, 0xbdc3c2c0, 0x77ba63e0, 0xfdf7b330,
    0xd7800df8, 0xedc0081c, 0xdfa0041a, 0x81d00a2d,

    0x80000000, 0x40000000, 0x20000000, 0x30000000,
    0x58000000, 0xac000000, 0x96000000, 0x2b000000,
    0xd4800000, 0x09400000, 0xe2a00000, 0x52500000,
    0x4e280000, 0xc71c0000, 0x629e0000, 0x12670000,
    0x6e138000, 0xf731c000, 0x3a98a000, 0xbe449000,
    0xf83b8800, 0xdc2dc400, 0xee06a200, 0xb7239300,
    0x1aa80d80, 0x8e5c0ec0, 0xa03e0b60, 0x703701b0,
    0x783b88c8, 0x9c2dca54, 0xce06a74a, 0x87239795,

    0x80000000, 0xc0000000, 0xa0000000, 0x50000000,
    0xf8000000, 0x8c000000, 0xe2000000, 0x33000000,
    0x0f800000, 0x21400000, 0x95a00000, 0x5e700000,
    0xd8080000, 0x1c240000, 0xba160000, 0xef370000,
    0x15868000, 0x9e6fc000, 0x781b6000, 0x4c349000,
    0x420e8800, 0x630bcc00, 0xf7ad6a00, 0xad739500,
    0x77800780, 0x6d4004c0, 0xd7a00420, 0x3d700630,
    0x2f880f78, 0xb1640ad4, 0xcdb6077a, 0x824706d7,

    0x80000000, 0xc0000000, 0x60000000, 0x90000000,
    0x38000000, 0xc4000000, 0x42000000, 0xa3000000,
    0xf1800000, 0xaa400000, 0xfce00000, 0x85100000,
    0xe0080000, 0x500c0000, 0x58060000, 0x54090000,
    0x7a038000, 0x670c4000, 0xb3842000, 0x094a3000,
    0x0d6f1800, 0x2f5aa400, 0x1ce7ce00, 0xd5145100,
    0xb8000080, 0x040000c0, 0x22000060, 0x33000090,
    0xc9800038, 0x6e4000c4, 0xbee00042, 0x261000a3,

    0x80000000, 0x40000000, 0x20000000, 0xf0000000,
    0xa8000000, 0x54000000, 0x9a000000, 0x9d000000,
    0x1e800000, 0x5cc00000, 0x7d200000, 0x8d100000,
    0x24880000, 0x71c40000, 0xeba20000, 0x75df0000,
    0x6ba28000, 0x35d14000, 0x4ba3a000, 0xc5d2d000,
    0xe3a16800, 0x91db8c00, 0x79aef200, 0x0cdf4100,
    0x672a8080, 0x50154040, 0x1a01a020, 0xdd0dd0f0,
    0x3e83e8a8, 0xaccacc54, 0xd52d529a, 0xd91d919d,

    0x80000000, 0xc0000000, 0x20000000, 0xd0000000,
    0xd8000000, 0xc4000000, 0x46000000, 0x85000000,
    0xa5800000, 0x76c00000, 0xada00000, 0x6ab00000,
    0x2da80000, 0xaabc0000, 0x0daa0000, 0x7ab10000,
    0xd5a78000, 0xbebd4000, 0x93a3e000, 0x3bb51000,
    0x3629b800, 0x4d727c00, 0x9b836200, 0x27c4d700,
    0xb629b880, 0x8d727cc0, 0xbb836220, 0xf7c4d7d0,
    0x6e29b858, 0x49727c04, 0xfd836266, 0x72c4d755
};

// =============================================================================
// Teaser Figure Matrices
// =============================================================================

const uint32_t SZTeaserFigColMatrices[16*32] = {
    0x80000000, 0x40000000, 0x20000000, 0x10000000,
    0x08000000, 0x04000000, 0x02000000, 0x01000000,
    0x00800000, 0x00400000, 0x00200000, 0x00100000,
    0x00080000, 0x00040000, 0x00020000, 0x00010000,
    0x00008000, 0x00004000, 0x00002000, 0x00001000,
    0x00000800, 0x00000400, 0x00000200, 0x00000100,
    0x00000080, 0x00000040, 0x00000020, 0x00000010,
    0x00000008, 0x00000004, 0x00000002, 0x00000001,

    0x80000000, 0xc0000000, 0xa0000000, 0xf0000000,
    0x88000000, 0xcc000000, 0xaa000000, 0xff000000,
    0x80800000, 0xc0c00000, 0xa0a00000, 0xf0f00000,
    0x88880000, 0xcccc0000, 0xaaaa0000, 0xffff0000,
    0x80008000, 0xc000c000, 0xa000a000, 0xf000f000,
    0x88008800, 0xcc00cc00, 0xaa00aa00, 0xff00ff00,
    0x80808080, 0xc0c0c0c0, 0xa0a0a0a0, 0xf0f0f0f0,
    0x88888888, 0xcccccccc, 0xaaaaaaaa, 0xffffffff,

    0x80000000, 0x40000000, 0x60000000, 0xd0000000,
    0xc8000000, 0x84000000, 0xb6000000, 0x6d000000,
    0x40800000, 0xc0400000, 0xd0600000, 0xb0d00000,
    0x84c80000, 0x4c840000, 0x6db60000, 0xdb6d0000,
    0xc0008000, 0x80004000, 0xb0006000, 0x6000d000,
    0x4c00c800, 0xc8008400, 0xdb00b600, 0xb6006d00,
    0x80c04080, 0x4080c040, 0x60b0d060, 0xd060b0d0,
    0xc84c84c8, 0x84c84c84, 0xb6db6db6, 0x6db6db6d,

    0x80000000, 0xc0000000, 0x60000000, 0xb0000000,
    0xc8000000, 0x4c000000, 0xb6000000, 0xdb000000,
    0x40800000, 0x80c00000, 0xd0600000, 0x60b00000,
    0x84c80000, 0xc84c0000, 0x6db60000, 0xb6db0000,
    0xc0008000, 0x4000c000, 0xb0006000, 0xd000b000,
    0x4c00c800, 0x84004c00, 0xdb00b600, 0x6d00db00,
    0x80c04080, 0xc04080c0, 0x60b0d060, 0xb0d060b0,
    0xc84c84c8, 0x4c84c84c, 0xb6db6db6, 0xdb6db6db,

    0x80000000, 0x40000000, 0x20000000, 0x10000000,
    0x18000000, 0x34000000, 0xa2000000, 0x51000000,
    0x50800000, 0xf0400000, 0xb0200000, 0x60100000,
    0x65180000, 0xdf340000, 0xeba20000, 0x96510000,
    0x90008000, 0x70004000, 0x80002000, 0x40001000,
    0x49001800, 0xc7003400, 0x1800a200, 0x34005100,
    0x30905080, 0x2070f040, 0x5080b020, 0xf0406010,
    0xf3496518, 0xa2c7df34, 0x6518eba2, 0xdf349651,

    0x80000000, 0xc0000000, 0xa0000000, 0xf0000000,
    0x78000000, 0x9c000000, 0x8a000000, 0xcf000000,
    0xb0800000, 0xd0c00000, 0x70a00000, 0x90f00000,
    0x2b780000, 0x3d9c0000, 0xb78a0000, 0xd9cf0000,
    0xf0008000, 0x5000c000, 0x2000a000, 0x3000f000,
    0xcf007800, 0x45009c00, 0xf2008a00, 0x5300cf00,
    0x90f0b080, 0xe050d0c0, 0xc02070a0, 0x403090f0,
    0xd9cf2b78, 0x6e453d9c, 0x9cf2b78a, 0xe453d9cf,

    0x80000000, 0x40000000, 0x60000000, 0xd0000000,
    0x18000000, 0x34000000, 0x56000000, 0xfd000000,
    0xd0800000, 0xb0400000, 0xe0600000, 0x90d00000,
    0xfd180000, 0xab340000, 0x4e560000, 0xc9fd0000,
    0x90008000, 0x70004000, 0x30006000, 0x2000d000,
    0xc9001800, 0x87003400, 0xb3005600, 0x6200fd00,
    0x2090d080, 0x1070b040, 0xa030e060, 0x502090d0,
    0x62c9fd18, 0xd187ab34, 0x7ab34e56, 0xe562c9fd,

    0x80000000, 0xc0000000, 0x60000000, 0xb0000000,
    0x78000000, 0x9c000000, 0x56000000, 0xab000000,
    0x30800000, 0x10c00000, 0x80600000, 0xc0b00000,
    0xd3780000, 0x619c0000, 0x78560000, 0x9cab0000,
    0xf0008000, 0x5000c000, 0x30006000, 0x1000b000,
    0x4f007800, 0x85009c00, 0xd3005600, 0x6100ab00,
    0xe0f03080, 0x705010c0, 0xf0308060, 0x5010c0b0,
    0x2e4fd378, 0x3785619c, 0x4fd37856, 0x85619cab,

    0x40000000, 0xc0000000, 0x10000000, 0x30000000,
    0x94000000, 0x7c000000, 0x81000000, 0x43000000,
    0x60400000, 0xd0c00000, 0xe0100000, 0x90300000,
    0x56940000, 0xfd7c0000, 0xbe810000, 0x69430000,
    0x10004000, 0x3000c000, 0xa0001000, 0x50003000,
    0x81009400, 0x43007c00, 0x2a008100, 0x15004300,
    0xe0106040, 0x9030d0c0, 0xc0a0e010, 0x80509030,
    0xbe815694, 0x6943fd7c, 0x7c2abe81, 0xe8156943,

    0xc0000000, 0x40000000, 0xf0000000, 0x50000000,
    0xbc000000, 0xd4000000, 0x7f000000, 0x95000000,
    0xa0c00000, 0xf0400000, 0x10f00000, 0x20500000,
    0xeabc0000, 0x7fd40000, 0x417f0000, 0x82950000,
    0x3000c000, 0x10004000, 0xd000f000, 0x60005000,
    0xc300bc00, 0x4100d400, 0xfd007f00, 0x56009500,
    0xb030a0c0, 0xd010f040, 0x70d010f0, 0x90602050,
    0xabc3eabc, 0xfd417fd4, 0x17fd417f, 0x29568295,

    0x40000000, 0xc0000000, 0xd0000000, 0xb0000000,
    0xd4000000, 0xbc000000, 0xed000000, 0x9b000000,
    0xe0400000, 0x90c00000, 0x10d00000, 0x30b00000,
    0x1ed40000, 0x39bc0000, 0x51ed0000, 0xf39b0000,
    0x50004000, 0xf000c000, 0x8000d000, 0x4000b000,
    0x8500d400, 0x4f00bc00, 0x6800ed00, 0xd4009b00,
    0x6050e040, 0xd0f090c0, 0x708010d0, 0xe04030b0,
    0x76851ed4, 0xed4f39bc, 0x276851ed, 0x1ed4f39b,

    0xc0000000, 0x40000000, 0xb0000000, 0xd0000000,
    0x7c000000, 0x94000000, 0x5b000000, 0xad000000,
    0x20c00000, 0x30400000, 0x40b00000, 0x80d00000,
    0x627c0000, 0xb3940000, 0x945b0000, 0xe8ad0000,
    0xf000c000, 0x50004000, 0x3000b000, 0x1000d000,
    0xcf007c00, 0x45009400, 0xb3005b00, 0xd100ad00,
    0x70f020c0, 0x90503040, 0x503040b0, 0xa01080d0,
    0x27cf627c, 0x3945b394, 0x45b3945b, 0x8ad1e8ad,

    0x40000000, 0xc0000000, 0x10000000, 0x30000000,
    0xa4000000, 0x5c000000, 0xd1000000, 0xb3000000,
    0x90400000, 0x70c00000, 0x80100000, 0x40300000,
    0x29a40000, 0x175c0000, 0xf8d10000, 0xa4b30000,
    0x60004000, 0xd000c000, 0xe0001000, 0x90003000,
    0xc600a400, 0x8d005c00, 0x3e00d100, 0x2900b300,
    0x50609040, 0xf0d070c0, 0xb0e08010, 0x60904030,
    0x75c629a4, 0xef8d175c, 0x4b3ef8d1, 0xc629a4b3,

    0xc0000000, 0x40000000, 0xf0000000, 0x50000000,
    0xec000000, 0x74000000, 0x4f000000, 0x85000000,
    0xb0c00000, 0xd0400000, 0x70f00000, 0x90500000,
    0x3bec0000, 0x1d740000, 0xd74f0000, 0x69850000,
    0xa000c000, 0xf0004000, 0x1000f000, 0x20005000,
    0xca00ec00, 0x4f007400, 0xf1004f00, 0x52008500,
    0xe0a0b0c0, 0x70f0d040, 0x401070f0, 0x80209050,
    0xbeca3bec, 0xd74f1d74, 0x74f1d74f, 0x98526985,

    0x40000000, 0xc0000000, 0xd0000000, 0xb0000000,
    0x64000000, 0xdc000000, 0x7d000000, 0xeb000000,
    0x90400000, 0x70c00000, 0x30d00000, 0x20b00000,
    0x19640000, 0x37dc0000, 0x537d0000, 0xf2eb0000,
    0xa0004000, 0x5000c000, 0xc000d000, 0x8000b000,
    0x4a006400, 0xc500dc00, 0xdc007d00, 0xb800eb00,
    0x60a09040, 0xd05070c0, 0x70c030d0, 0xe08020b0,
    0x964a1964, 0x7dc537dc, 0x37dc537d, 0x2eb8f2eb,

    0xc0000000, 0x40000000, 0xb0000000, 0xd0000000,
    0xac000000, 0xf4000000, 0x2b000000, 0x3d000000,
    0xb0c00000, 0xd0400000, 0xe0b00000, 0x70d00000,
    0x2bac0000, 0x3df40000, 0x4e2b0000, 0x873d0000,
    0xe000c000, 0x70004000, 0xf000b000, 0x5000d000,
    0x4e00ac00, 0x8700f400, 0xdf002b00, 0x65003d00,
    0xf0e0b0c0, 0x5070d040, 0x30f0e0b0, 0x105070d0,
    0xdf4e2bac, 0x65873df4, 0x73df4e2b, 0x9165873d
};


#endif                                                                          // #ifndef SZ_H
